package au.com.fincleartech.ibroker;

import com.google.protobuf.InvalidProtocolBufferException;
import com.google.rpc.BadRequest;
import com.google.rpc.Code;
import io.fincleartech.protobuf.ibroker.accounts.CreateFinancialSettlementDetailsRequest;
import io.fincleartech.protobuf.ibroker.accounts.CreateFinancialSettlementDetailsResponse;
import io.fincleartech.protobuf.ibroker.accounts.IBrokerAccountsServiceGrpc;
import io.fincleartech.protobuf.ibroker.settlement.DemandTransferConfirmation;
import io.fincleartech.protobuf.ibroker.subscribe.IBrokerAcknowledge;
import io.fincleartech.protobuf.ibroker.subscribe.IBrokerMessage;
import io.fincleartech.protobuf.ibroker.subscribe.IBrokerSubscribeServiceGrpc;
import io.grpc.ManagedChannel;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import io.grpc.netty.shaded.io.grpc.netty.GrpcSslContexts;
import io.grpc.netty.shaded.io.grpc.netty.NettyChannelBuilder;
import io.grpc.netty.shaded.io.netty.handler.ssl.SslContext;
import io.grpc.stub.StreamObserver;

import javax.net.ssl.SSLException;
import java.io.File;
import java.util.UUID;
import java.util.function.Function;

@SuppressWarnings({"ResultOfMethodCallIgnored", "WeakerAccess", "StatementWithEmptyBody"})
public class Example {

    private StreamObserver<IBrokerAcknowledge> subscription = null;

    public static void main(String[] args) throws SSLException, InvalidProtocolBufferException {
        Example example = new Example();
        example.go(args);
    }

    public void go(String[] args) throws SSLException, InvalidProtocolBufferException {
        // setup SSL/authentication
        SslContext sslContext = GrpcSslContexts.forClient()
                .keyManager(new File("cert.pem"), new File("key.pem"))
                .trustManager(new File("ca.pem"))
                .build();

        // build channel
        ManagedChannel managedChannel = NettyChannelBuilder.forAddress(args[0], Integer.parseInt(args[1]))
                .sslContext(sslContext)
                .build();

        // streaming stub for listening to changes
        IBrokerSubscribeServiceGrpc.IBrokerSubscribeServiceStub iBrokerStreaming = IBrokerSubscribeServiceGrpc.newStub(managedChannel);

        // subscribe to changes
        subscription = iBrokerStreaming.subscribe(new StreamObserver<IBrokerMessage>() {
            @Override
            public void onNext(IBrokerMessage message) {
                switch (message.getMessageTypeCase()) {
                    case DEMAND_TRANSFER_CONFIRMATION:
                        DemandTransferConfirmation demandTransferConfirmation = message.getDemandTransferConfirmation();
                        // process
                        // ...
                        // acknowledge message
                        IBrokerAcknowledge ack = IBrokerAcknowledge.newBuilder()
                                .setMessageId(message.getMessageId())
                                .build();
                        subscription.onNext(ack);
                        break;
                    case MESSAGETYPE_NOT_SET:
                    default:
                        break;
                }
            }

            @Override
            public void onError(Throwable t) {
            }

            @Override
            public void onCompleted() {
            }
        });

        // synchronous stub for making requests
        IBrokerAccountsServiceGrpc.IBrokerAccountsServiceBlockingStub iBroker = IBrokerAccountsServiceGrpc.newBlockingStub(managedChannel);

        // request
        CreateFinancialSettlementDetailsRequest.Builder requestBuilder = CreateFinancialSettlementDetailsRequest.newBuilder();
        requestBuilder.getRequestIdBuilder().setValue(UUID.randomUUID().toString());
        requestBuilder.getAccountNumberBuilder().setValue("10040");
        requestBuilder.getBsbNumberBuilder().setValue("013006");
        requestBuilder.getBankAccountNumberBuilder().setValue("940010037");
        requestBuilder.getBankAccountNameBuilder().setValue("EVE MAY WRIGHT");
        requestBuilder.getDirectDebitBuilder().setValue(false);
        requestBuilder.getDefaultPaymentBuilder().setValue(false);
        requestBuilder.getDefaultReceiptBuilder().setValue(false);
        requestBuilder.getCurrentBuilder().setValue(true);
        requestBuilder.getCmtFundBuilder().setValue("");
        requestBuilder.getPaymentGenMethodBuilder().setValue("A");
        requestBuilder.getReceiptGenMethodBuilder().setValue("D");
        requestBuilder.getSettlementTypeBuilder().setValue("T");
        requestBuilder.getAutoContraBuilder().setValue(true);
        requestBuilder.getSettlementGroupBuilder().setValue("GEN");
        requestBuilder.getHoldReasonBuilder().setValue("");

        try {
            CreateFinancialSettlementDetailsResponse response = iBroker.createFinancialSettlementDetails(requestBuilder.build());
            com.google.rpc.Status status = response.getStatus();
            Code statusCode = Code.forNumber(status.getCode());
            if (statusCode == Code.OK) {
                String accountNumber = response.getAccountNumber().getValue();
                String financialSettlementDetailsId = response.getFinancialSettlementDetailsId().getValue();
            } else {
                // Business/Application level errors.
                String errorMessage = status.getMessage();
                System.out.println(errorMessage);

                if (statusCode == Code.INVALID_ARGUMENT) {
                    status.getDetailsList().stream()
                            .filter(any -> any.is(BadRequest.class))
                            .map(Exceptions.wrap(any -> any.unpack(BadRequest.class)))
                            .flatMap(badRequest -> badRequest.getFieldViolationsList().stream())
                            .forEach(violation -> System.out.println(violation.getField() + ": " + violation.getDescription()));
                } else if (statusCode == Code.ALREADY_EXISTS) {
                    // request already in process
                }
            }
        } catch (StatusRuntimeException e) {
            // Protocol level errors.
            Status status = e.getStatus();
            switch (status.getCode()) {
                case CANCELLED:
                    break;
                case UNKNOWN:
                    break;
                case PERMISSION_DENIED:
                    break;
                case UNIMPLEMENTED:
                    break;
                case UNAVAILABLE:
                    break;
                case UNAUTHENTICATED:
                    break;
            }
        }


    }

    private static class Exceptions {
        public interface EFn<T, R, E extends Exception> {
            R apply(T t) throws E;
        }

        public static <T, R, E extends Exception> Function<T, R> wrap(EFn<T, R, E> f) {
            return t -> {
                try {
                    return f.apply(t);
                } catch (Exception e) {
                    if (e instanceof RuntimeException) {
                        throw (RuntimeException) e;
                    }
                    throw new RuntimeException(e);
                }
            };
        }
    }
}
